<?php

namespace App\Http\Controllers;

use App\Models\Address;
use Illuminate\View\View;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Http\RedirectResponse;
use Illuminate\Support\Facades\Redirect;
use App\Http\Requests\ProfileUpdateRequest;
use Intervention\Image\Colors\Rgb\Channels\Red;

class ProfileController extends Controller
{
    /**
     * Display the user's profile form.
     */
    public function index(Request $request): View
    {
        return view('profile.index', [
            'user' => $request->user(),
        ]);
    }

    public function edit(Request $request): View
    {
        return view('profile.index', [
            'user' => $request->user(),
        ]);
    }

    /**
     * Update the user's profile information.
     */
    public function update(ProfileUpdateRequest $request): RedirectResponse
    {
        $request->user()->fill($request->validated());

        if ($request->user()->isDirty('email')) {
            $request->user()->email_verified_at = null;
        }

        $request->user()->save();

        return Redirect::route('profile.edit', user()->id)->with('status', 'profile-updated');
    }

    /**
     * Update the user's password.
     */
    public function password(Request $request): RedirectResponse
    {
        $request->validate([
            'current_password' => ['required', 'current_password'],
            'password' => ['required', 'confirmed', 'min:8'],
        ]);

        $request->user()->update([
            'password' => bcrypt($request->password),
        ]);

        return Redirect::route('profile.edit', user()->id)->with('status', 'password-updated');
    }

    /**
     * Delete the user's account.
     */
    public function destroy(Request $request): RedirectResponse
    {
        $request->validateWithBag('userDeletion', [
            'password' => ['required', 'current_password'],
        ]);

        $user = $request->user();

        Auth::logout();

        $user->delete();

        $request->session()->invalidate();
        $request->session()->regenerateToken();

        return Redirect::to('/');
    }

    // Address
    public function address(Request $request)
    {
        return view('profile.address', [
            'user' => $request->user(),
            'address' => $request->user()->address?? new Address(),
        ]);
    }

    public function addressUpdate(Request $request): RedirectResponse
    {
        $data = $request->validate([
            'address' => ['required', 'string'],
            'address_one' => ['nullable', 'string'],
            'address_two' => ['nullable', 'string'],
            'city_id' => ['required', 'integer', 'exists:cities,id'],
            'state_id' => ['required', 'integer', 'exists:states,id'],
            'country_id' => ['required', 'integer', 'exists:countries,id'],
            'postal' => ['required', 'string'],
            'landmark' => ['nullable', 'string'],
            'latitude' => ['nullable', 'numeric'],
            'longitude' => ['nullable', 'numeric'],
            'is_default' => ['required', 'boolean'],
        ]);

        $request->user()->address()->updateOrCreate(
            ['user_id' => $request->user()->id],
            $data
        );

        return Redirect::route('profile.address')->with('success', 'Address updated successfully');
    }
}
